#!/usr/bin/env bash
# Usage: ./patient_inotify.sh events watched_file interval_seconds command [args]
# Description: Run 'command' every time 'events' happen to 'watched_file'.
#              If they happen faster than 'interval_seconds' then wait at
#              least that amount before executing the command.

# exit on error
set -e

if [ $# -lt 4 ]; then
    echo "Usage: ./patient_inotify.sh events watched_file interval_seconds command [args]"
    exit 1
fi

current_seconds() {
    date +%s
}

log() {
    echo "INFO: `date +%Y-%m-%dT%H:%M:%S` : $@" >&2
}

events=$1
watched_file=$2
interval_seconds=$3
command=$4
# make it so $@ is the remaining arguments
shift 4

next_seconds=0
last_event=`current_seconds`
timeout=0
while true; do
    inotifywait -e "$events" -t $timeout -qq --format "%T,%f,%:e" --timefmt "%s" "$watched_file" || log "Timeout reached"

    # if events are happening too often
    curr_seconds=`current_seconds`
    if [ $next_seconds -gt $curr_seconds ]; then
        # events are coming in faster than we want them
        # wait until it has been enough time but not more than interval
        timeout=$interval_seconds
        continue
    fi

    # run the command for this event
    log "Running command..."
    $command $@
    next_seconds=`current_seconds`
    next_seconds=$(($next_seconds + $interval_seconds))
    timeout=0
done
